<?php
/**
 * Plugin Name: Payment Stats
 * Description: Estadísticas de métodos de pago para WooCommerce.
 * Version: 1.1
 * Author: Maikel-J
 */

// Evitar acceso directo al archivo
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Registrar el menú en el backend
add_action( 'admin_menu', 'payment_stats_menu' );

function payment_stats_menu() {
    add_menu_page(
        'Payment Stats',
        'Payment Stats',
        'manage_woocommerce',
        'payment-stats',
        'payment_stats_page',
        'dashicons-chart-bar',
        56
    );
}

// Crear la página del plugin
function payment_stats_page() {
    if ( ! class_exists( 'WooCommerce' ) ) {
        echo '<div class="notice notice-error"><p>WooCommerce no está activado.</p></div>';
        return;
    }

    $start_date = isset( $_GET['start_date'] ) ? sanitize_text_field( $_GET['start_date'] ) . ' 00:00:00' : date( 'Y-m-01 00:00:00' );
    $end_date   = isset( $_GET['end_date'] ) ? sanitize_text_field( $_GET['end_date'] ) . ' 23:59:59' : date( 'Y-m-d 23:59:59' );

    global $wpdb;

    $query = $wpdb->prepare(
        "
        SELECT pm.meta_value AS payment_method, 
               COUNT(*) AS number_of_orders, 
               SUM(o.meta_value) AS money_received
        FROM {$wpdb->postmeta} pm
        INNER JOIN {$wpdb->postmeta} o ON pm.post_id = o.post_id
        WHERE pm.meta_key = '_payment_method'
          AND o.meta_key = '_order_total'
          AND pm.post_id IN (
              SELECT ID FROM {$wpdb->posts} 
              WHERE post_type = 'shop_order' 
                AND post_status IN ('wc-completed', 'wc-processing')
                AND post_date BETWEEN %s AND %s
          )
        GROUP BY pm.meta_value
        ",
        $start_date, $end_date
    );

    $results = $wpdb->get_results( $query );

    ?>
    <style>
	.payment-stats-pagination {
    margin-top: 20px;
    text-align: center;
}
.payment-stats-pagination .page-numbers {
    display: inline-block;
    padding: 8px 14px;
    margin: 0 3px;
    border: 1px solid #ccd0d4;
    border-radius: 5px;
    background-color: #fff;
    color: #007cba;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.2s ease-in-out;
}
.payment-stats-pagination .page-numbers:hover {
    background-color: #007cba;
    color: #fff;
}
.payment-stats-pagination .page-numbers.current {
    background-color: #007cba;
    color: #fff;
    border-color: #007cba;
    pointer-events: none;
}

        .payment-stats-form {
            background: #fff;
            padding: 20px;
            border: 1px solid #ccd0d4;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin-bottom: 20px;
            display: inline-block;
        }
        .payment-stats-form label {
            margin-right: 10px;
            font-weight: bold;
        }
        .payment-stats-form input[type="date"] {
            margin-right: 10px;
            padding: 5px 8px;
            border-radius: 4px;
            border: 1px solid #ccd0d4;
        }
        .payment-stats-form .button-primary {
            background: #007cba;
            border-color: #007cba;
            color: #fff;
            padding: 6px 12px;
            border-radius: 4px;
            text-transform: uppercase;
            font-weight: bold;
        }
        .payment-stats-form .button-primary:hover {
            background: #006ba1;
            border-color: #006ba1;
        }
        .payment-stats-table {
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
            border: 1px solid #ccd0d4;
            border-radius: 8px;
            overflow: hidden;
        }
        .payment-stats-table thead {
            background-color: #f7f7f7;
        }
        .payment-stats-table th, .payment-stats-table td {
            padding: 12px 15px;
            text-align: left;
        }
        .payment-stats-table tbody tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        .payment-stats-table tbody tr:hover {
            background-color: #eef6fb;
        }
    </style>
    <div class="wrap">
        <h1>Estadísticas de Métodos de Pago</h1>
        <form method="get" class="payment-stats-form">
            <input type="hidden" name="page" value="payment-stats">
            <label for="start_date">Fecha Inicio:</label>
            <input type="date" id="start_date" name="start_date" value="<?php echo esc_attr( substr( $start_date, 0, 10 ) ); ?>">
            <label for="end_date">Fecha Fin:</label>
            <input type="date" id="end_date" name="end_date" value="<?php echo esc_attr( substr( $end_date, 0, 10 ) ); ?>">
            <button type="submit" class="button button-primary">Filtrar</button>
        </form>
        <table class="payment-stats-table">
            <thead>
                <tr>
                    <th>Método de Pago</th>
                    <th>Número de Órdenes</th>
                    <th>Dinero Total</th>
                </tr>
            </thead>
            <tbody>
                <?php if ( $results ) : ?>
                    <?php foreach ( $results as $row ) : ?>
                        <tr>
                            <?php
                            $gateways = WC()->payment_gateways()->get_available_payment_gateways();
                            $payment_title = isset( $gateways[ $row->payment_method ] ) ? $gateways[ $row->payment_method ]->get_title() : $row->payment_method;
                            ?>
                            <td><?php echo esc_html( $payment_title ); ?></td>
                            <td>
                                <a href="<?php echo admin_url( 'admin.php?page=payment-stats-orders&payment_method=' . urlencode( $row->payment_method ) . '&start_date=' . substr( $start_date, 0, 10 ) . '&end_date=' . substr( $end_date, 0, 10 ) ); ?>">
                                    <?php echo esc_html( $row->number_of_orders ); ?>
                                </a>
                            </td>
                            <td><?php echo wc_price( $row->money_received ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php else : ?>
                    <tr>
                        <td colspan="3">No se encontraron resultados para el rango seleccionado.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    <?php
}

// Página para mostrar las órdenes de un método de pago específico
add_action( 'admin_menu', 'payment_stats_orders_menu' );

function payment_stats_orders_menu() {
    add_submenu_page(
        null,
        'Orders by Payment Method',
        'Orders by Payment Method',
        'manage_woocommerce',
        'payment-stats-orders',
        'payment_stats_orders_page'
    );
}

function payment_stats_orders_page() {
    if ( ! class_exists( 'WooCommerce' ) ) {
        echo '<div class="notice notice-error"><p>WooCommerce no está activado.</p></div>';
        return;
    }

    $payment_method = isset( $_GET['payment_method'] ) ? sanitize_text_field( $_GET['payment_method'] ) : '';
    $start_date = isset( $_GET['start_date'] ) ? sanitize_text_field( $_GET['start_date'] ) . ' 00:00:00' : '';
    $end_date   = isset( $_GET['end_date'] ) ? sanitize_text_field( $_GET['end_date'] ) . ' 23:59:59' : '';

    global $wpdb;

    $per_page = 20;
    $current_page = isset( $_GET['paged'] ) ? max( 1, intval( $_GET['paged'] ) ) : 1;
    $offset = ( $current_page - 1 ) * $per_page;

    $total_orders = $wpdb->get_var( $wpdb->prepare( "
        SELECT COUNT(*)
        FROM {$wpdb->posts}
        WHERE ID IN (
            SELECT post_id
            FROM {$wpdb->postmeta}
            WHERE meta_key = '_payment_method'
              AND meta_value = %s
        )
        AND post_date BETWEEN %s AND %s
        AND post_type = 'shop_order'
        AND post_status IN ('wc-completed', 'wc-processing')
    ", $payment_method, $start_date, $end_date ) );

    $orders = $wpdb->get_results( $wpdb->prepare( "
        SELECT ID, post_date, post_status
        FROM {$wpdb->posts}
        WHERE ID IN (
            SELECT post_id
            FROM {$wpdb->postmeta}
            WHERE meta_key = '_payment_method'
              AND meta_value = %s
        )
        AND post_date BETWEEN %s AND %s
        AND post_type = 'shop_order'
        AND post_status IN ('wc-completed', 'wc-processing')
        ORDER BY post_date DESC
        LIMIT %d OFFSET %d
    ", $payment_method, $start_date, $end_date, $per_page, $offset ) );

    $gateways = WC()->payment_gateways()->get_available_payment_gateways();
    $payment_title = isset( $gateways[ $payment_method ] ) ? $gateways[ $payment_method ]->get_title() : $payment_method;

    ?>
    <style>
	     .payment-stats-pagination {
    margin-top: 20px;
    text-align: center;
}
.payment-stats-pagination .page-numbers {
    display: inline-block;
    padding: 8px 14px;
    margin: 0 3px;
    border: 1px solid #ccd0d4;
    border-radius: 5px;
    background-color: #fff;
    color: #007cba;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.2s ease-in-out;
}
.payment-stats-pagination .page-numbers:hover {
    background-color: #007cba;
    color: #fff;
}
.payment-stats-pagination .page-numbers.current {
    background-color: #007cba;
    color: #fff;
    border-color: #007cba;
    pointer-events: none;
}

        .payment-stats-table {
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
            border: 1px solid #ccd0d4;
            border-radius: 8px;
            overflow: hidden;
        }
        .payment-stats-table thead {
            background-color: #f7f7f7;
        }
        .payment-stats-table th, .payment-stats-table td {
            padding: 12px 15px;
            text-align: left;
        }
        .payment-stats-table tbody tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        .payment-stats-table tbody tr:hover {
            background-color: #eef6fb;
        }
    </style>
    <div class="wrap">
        <h1>Órdenes con <?php echo esc_html( $payment_title ); ?> (<?php echo esc_html( $total_orders ); ?> órdenes)</h1>

        <table class="payment-stats-table">
            <thead>
                <tr>
                    <th>ID de Orden</th>
                    <th>Fecha</th>
                    <th>Estado</th>
                </tr>
            </thead>
            <tbody>
                <?php if ( $orders ) : ?>
                    <?php foreach ( $orders as $order ) : ?>
                        <tr>
                            <td><a href="<?php echo get_edit_post_link( $order->ID ); ?>">#<?php echo esc_html( $order->ID ); ?></a></td>
                            <td><?php echo esc_html( $order->post_date ); ?></td>
                            <td><?php echo esc_html( $order->post_status ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php else : ?>
                    <tr>
                        <td colspan="3">No se encontraron órdenes para este método de pago.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
        <?php
        $pagination_args = array(
            'total'   => ceil( $total_orders / $per_page ),
            'current' => $current_page,
            'base'    => add_query_arg( 'paged', '%#%' ),
            'format'  => '',
        );
        echo '<div class="payment-stats-pagination">' . paginate_links( $pagination_args ) . '</div>';

        ?>
    </div>
    <?php
}
