<?php
/**
 * Plugin Name: Payment Stats
 * Description: Estadísticas de métodos de pago y referrals para WooCommerce.
 * Version: 1.2.0
 * Author: Maikel-J (+referrals)
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/* ========= CONFIG META KEYS (ajustadas a tu instalación) =========
   _wc_order_attribution_source_type = 'referral'
   _wc_order_attribution_referrer    = URL del referente (ej. https://www.idealo.de/)
*/
const PS_META_REFERRER_KEY   = '_wc_order_attribution_referrer';
const PS_META_SOURCE_TYPE_KEY= '_wc_order_attribution_source_type';

/* ========= ADMIN: Enqueue datepicker para mostrar calendarios siempre ========= */
add_action('admin_enqueue_scripts', function($hook){
    // Solo en la página principal del plugin (evita cargar en todo el admin)
    if ( $hook !== 'toplevel_page_payment-stats' ) return;
    wp_enqueue_script('jquery-ui-datepicker');
    wp_enqueue_style('jquery-ui-theme', 'https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css', [], '1.13.2');
    wp_add_inline_script('jquery-ui-datepicker', "
        jQuery(function($){
            $('.ps-date').datepicker({
                dateFormat:'yy-mm-dd',
                changeMonth:true,
                changeYear:true
            });
        });
    ");
});

/* ========= MENÚ ========= */
add_action( 'admin_menu', function () {
    add_menu_page(
        'Payment/Referral Stats',
        'Payment Stats',
        'manage_woocommerce',
        'payment-stats',
        'payment_stats_page',
        'dashicons-chart-bar',
        56
    );
    // Subpáginas ocultas para listados
    add_submenu_page( null, 'Orders by Payment Method', 'Orders by Payment Method', 'manage_woocommerce', 'payment-stats-orders', 'payment_stats_orders_page' );
    add_submenu_page( null, 'Orders by Referral',       'Orders by Referral',       'manage_woocommerce', 'payment-referrals-orders', 'payment_referrals_orders_page' );
});

/* ========= PÁGINA PRINCIPAL (tabs) ========= */
function payment_stats_page() {
    if ( ! class_exists( 'WooCommerce' ) ) {
        echo '<div class="notice notice-error"><p>WooCommerce no está activado.</p></div>';
        return;
    }

    $view       = isset($_GET['view']) ? sanitize_key($_GET['view']) : 'payments';
    $start_date = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']).' 00:00:00' : date('Y-m-01 00:00:00');
    $end_date   = isset($_GET['end_date'])   ? sanitize_text_field($_GET['end_date']).' 23:59:59'   : date('Y-m-d 23:59:59');

    ?>
    <style>
    .ps-tabs{margin:0 0 10px;display:flex;gap:8px}
    .ps-tabs a{padding:6px 10px;border:1px solid #ccd0d4;border-radius:6px;text-decoration:none}
    .ps-tabs a.active{background:#007cba;color:#fff;border-color:#007cba}
    .payment-stats-form{background:#fff;padding:20px;border:1px solid #ccd0d4;border-radius:8px;box-shadow:0 2px 4px rgba(0,0,0,0.1);margin-bottom:20px;display:inline-block}
    .payment-stats-form label{margin-right:10px;font-weight:700}
    .payment-stats-form input[type="text"]{margin-right:10px;padding:5px 8px;border-radius:4px;border:1px solid #ccd0d4}
    .payment-stats-form .button-primary{background:#007cba;border-color:#007cba;color:#fff;padding:6px 12px;border-radius:4px;text-transform:uppercase;font-weight:bold}
    .payment-stats-form .button-primary:hover{background:#006ba1;border-color:#006ba1}
    .payment-stats-table{border-collapse:separate;border-spacing:0;width:100%;border:1px solid #ccd0d4;border-radius:8px;overflow:hidden}
    .payment-stats-table thead{background:#f7f7f7}
    .payment-stats-table th,.payment-stats-table td{padding:12px 15px;text-align:left}
    .payment-stats-table tbody tr:nth-child(even){background:#f9f9f9}
    .payment-stats-table tbody tr:hover{background:#eef6fb}
    .payment-stats-pagination{margin-top:20px;text-align:center}
    .payment-stats-pagination .page-numbers{display:inline-block;padding:8px 14px;margin:0 3px;border:1px solid #ccd0d4;border-radius:5px;background:#fff;color:#007cba;text-decoration:none;font-weight:600;transition:all .2s}
    .payment-stats-pagination .page-numbers:hover{background:#007cba;color:#fff}
    .payment-stats-pagination .page-numbers.current{background:#007cba;color:#fff;border-color:#007cba;pointer-events:none}
    </style>

    <div class="wrap">
        <h1>Estadísticas</h1>

        <div class="ps-tabs">
            <a class="<?php echo $view==='payments'?'active':''; ?>" href="<?php echo esc_url( add_query_arg(['page'=>'payment-stats','view'=>'payments','start_date'=>substr($start_date,0,10),'end_date'=>substr($end_date,0,10)]) ); ?>">Métodos de Pago</a>
            <a class="<?php echo $view==='referrals'?'active':''; ?>" href="<?php echo esc_url( add_query_arg(['page'=>'payment-stats','view'=>'referrals','start_date'=>substr($start_date,0,10),'end_date'=>substr($end_date,0,10)]) ); ?>">Referrals (Source)</a>
        </div>

        <form method="get" class="payment-stats-form">
            <input type="hidden" name="page" value="payment-stats">
            <input type="hidden" name="view" value="<?php echo esc_attr($view); ?>">
            <label for="start_date">Fecha Inicio:</label>
            <input type="text" id="start_date" class="ps-date" name="start_date" value="<?php echo esc_attr(substr($start_date,0,10)); ?>" placeholder="YYYY-MM-DD">
            <label for="end_date">Fecha Fin:</label>
            <input type="text" id="end_date" class="ps-date" name="end_date" value="<?php echo esc_attr(substr($end_date,0,10)); ?>" placeholder="YYYY-MM-DD">
            <button type="submit" class="button button-primary">Filtrar</button>
        </form>
    <?php

    if ( $view === 'referrals' ) {
        ps_render_referrals_table( $start_date, $end_date );
    } else {
        ps_render_payments_table( $start_date, $end_date );
    }

    echo '</div>';
}

/* ========= TABLA: MÉTODOS DE PAGO ========= */
function ps_render_payments_table( $start_date, $end_date ) {
    global $wpdb;

    $query = $wpdb->prepare("
        SELECT pm.meta_value AS payment_method,
               COUNT(*) AS number_of_orders,
               SUM(ot.meta_value) AS money_received
        FROM {$wpdb->postmeta} pm
        INNER JOIN {$wpdb->postmeta} ot 
                ON pm.post_id = ot.post_id AND ot.meta_key = '_order_total'
        WHERE pm.meta_key = '_payment_method'
          AND pm.post_id IN (
            SELECT ID FROM {$wpdb->posts}
            WHERE post_type='shop_order'
              AND post_status IN ('wc-completed','wc-processing')
              AND post_date BETWEEN %s AND %s
          )
        GROUP BY pm.meta_value
    ", $start_date, $end_date );

    $results = $wpdb->get_results( $query );

    echo '<table class="payment-stats-table"><thead><tr><th>Método de Pago</th><th>Número de Órdenes</th><th>Dinero Total</th></tr></thead><tbody>';

    if ( $results ) {
        $gateways = WC()->payment_gateways()->get_available_payment_gateways();
        foreach ( $results as $row ) {
            $payment_title = isset( $gateways[$row->payment_method] ) ? $gateways[$row->payment_method]->get_title() : $row->payment_method;
            $link = admin_url( 'admin.php?page=payment-stats-orders&payment_method=' . urlencode($row->payment_method) . '&start_date=' . substr($start_date,0,10) . '&end_date=' . substr($end_date,0,10) );
            echo '<tr>';
            echo '<td>'.esc_html($payment_title).'</td>';
            echo '<td><a href="'.esc_url($link).'">'.esc_html($row->number_of_orders).'</a></td>';
            echo '<td>'.wc_price($row->money_received).'</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="3">No se encontraron resultados.</td></tr>';
    }

    echo '</tbody></table>';
}

/* ========= TABLA: REFERRALS (agrupa por dominio del referrer) ========= */
function ps_render_referrals_table( $start_date, $end_date ) {
    global $wpdb;

    $ref_key  = esc_sql(PS_META_REFERRER_KEY);
    $type_key = esc_sql(PS_META_SOURCE_TYPE_KEY);

    // Dominio desde URL: elimina protocolo y toma host (antes del primer '/')
    $host_sql = "LOWER(SUBSTRING_INDEX(SUBSTRING_INDEX(pm_src.meta_value, '//', -1), '/', 1))";

    $query = $wpdb->prepare("
        SELECT {$host_sql} AS ref_host,
               COUNT(DISTINCT p.ID)                 AS number_of_orders,
               SUM(CAST(ot.meta_value AS DECIMAL(20,6))) AS money_received
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} ot
               ON ot.post_id = p.ID AND ot.meta_key = '_order_total'
        LEFT JOIN {$wpdb->postmeta} pm_src
               ON pm_src.post_id = p.ID AND pm_src.meta_key = '{$ref_key}'
        LEFT JOIN {$wpdb->postmeta} pm_typ
               ON pm_typ.post_id = p.ID AND pm_typ.meta_key = '{$type_key}' AND pm_typ.meta_value = 'referral'
        WHERE p.post_type = 'shop_order'
          AND p.post_status IN ('wc-completed','wc-processing')
          AND p.post_date BETWEEN %s AND %s
          AND pm_typ.post_id IS NOT NULL
          AND pm_src.meta_value <> ''
        GROUP BY ref_host
        ORDER BY money_received DESC
    ", $start_date, $end_date );

    $results = $wpdb->get_results( $query );

    echo '<table class="payment-stats-table"><thead><tr><th>Referral (Dominio)</th><th>Número de Órdenes</th><th>Dinero Total</th></tr></thead><tbody>';

    if ( $results ) {
        foreach ( $results as $row ) {
            $label = $row->ref_host ?: '(desconocido)';
            $link  = admin_url( 'admin.php?page=payment-referrals-orders&ref_source=' . urlencode($label) . '&start_date=' . substr($start_date,0,10) . '&end_date=' . substr($end_date,0,10) );
            echo '<tr>';
            echo '<td>'.esc_html($label).'</td>';
            echo '<td><a href="'.esc_url($link).'">'.esc_html($row->number_of_orders).'</a></td>';
            echo '<td>'.wc_price($row->money_received).'</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="3">No se encontraron referrals en el rango seleccionado.</td></tr>';
    }
    echo '</tbody></table>';
}

/* ========= LISTADO: ÓRDENES POR MÉTODO DE PAGO ========= */
function payment_stats_orders_page() {
    if ( ! class_exists( 'WooCommerce' ) ) { echo '<div class="notice notice-error"><p>WooCommerce no está activado.</p></div>'; return; }

    $payment_method = isset($_GET['payment_method']) ? sanitize_text_field($_GET['payment_method']) : '';
    $start_date     = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']).' 00:00:00' : '';
    $end_date       = isset($_GET['end_date'])   ? sanitize_text_field($_GET['end_date']).' 23:59:59'   : '';

    global $wpdb;
    $per_page = 20;
    $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
    $offset = ($current_page - 1) * $per_page;

    $total_orders = $wpdb->get_var( $wpdb->prepare("
        SELECT COUNT(*)
        FROM {$wpdb->posts}
        WHERE ID IN (
            SELECT post_id FROM {$wpdb->postmeta}
            WHERE meta_key='_payment_method' AND meta_value=%s
        )
        AND post_date BETWEEN %s AND %s
        AND post_type='shop_order'
        AND post_status IN ('wc-completed','wc-processing')
    ", $payment_method, $start_date, $end_date ) );

    $orders = $wpdb->get_results( $wpdb->prepare("
        SELECT ID, post_date, post_status
        FROM {$wpdb->posts}
        WHERE ID IN (
            SELECT post_id FROM {$wpdb->postmeta}
            WHERE meta_key='_payment_method' AND meta_value=%s
        )
        AND post_date BETWEEN %s AND %s
        AND post_type='shop_order'
        AND post_status IN ('wc-completed','wc-processing')
        ORDER BY post_date DESC
        LIMIT %d OFFSET %d
    ", $payment_method, $start_date, $end_date, $per_page, $offset ) );

    $gateways = WC()->payment_gateways()->get_available_payment_gateways();
    $payment_title = isset($gateways[$payment_method]) ? $gateways[$payment_method]->get_title() : $payment_method;

    echo '<style>.payment-stats-table{border-collapse:separate;border-spacing:0;width:100%;border:1px solid #ccd0d4;border-radius:8px;overflow:hidden}.payment-stats-table thead{background:#f7f7f7}.payment-stats-table th,.payment-stats-table td{padding:12px 15px;text-align:left}.payment-stats-table tbody tr:nth-child(even){background:#f9f9f9}.payment-stats-table tbody tr:hover{background:#eef6fb}.payment-stats-pagination{margin-top:20px;text-align:center}.payment-stats-pagination .page-numbers{display:inline-block;padding:8px 14px;margin:0 3px;border:1px solid #ccd0d4;border-radius:5px;background:#fff;color:#007cba;text-decoration:none;font-weight:600;transition:all .2s}.payment-stats-pagination .page-numbers:hover{background:#007cba;color:#fff}.payment-stats-pagination .page-numbers.current{background:#007cba;color:#fff;border-color:#007cba;pointer-events:none}</style>';

    echo '<div class="wrap"><h1>Órdenes con '.esc_html($payment_title).' ('.intval($total_orders).' órdenes)</h1>';
    echo '<table class="payment-stats-table"><thead><tr><th>ID de Orden</th><th>Fecha</th><th>Estado</th></tr></thead><tbody>';

    if ( $orders ) {
        foreach ( $orders as $o ) {
            echo '<tr>';
            echo '<td><a href="'.get_edit_post_link($o->ID).'">#'.esc_html($o->ID).'</a></td>';
            echo '<td>'.esc_html($o->post_date).'</td>';
            echo '<td>'.esc_html($o->post_status).'</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="3">No se encontraron órdenes para este método de pago.</td></tr>';
    }
    echo '</tbody></table>';

    $pagination_args = [
        'total'   => max(1, ceil($total_orders / $per_page)),
        'current' => $current_page,
        'base'    => add_query_arg('paged', '%#%'),
        'format'  => '',
    ];
    echo '<div class="payment-stats-pagination">'.paginate_links($pagination_args).'</div></div>';
}

/* ========= LISTADO: ÓRDENES POR REFERRAL (dominio) ========= */
function payment_referrals_orders_page() {
    if ( ! class_exists( 'WooCommerce' ) ) { echo '<div class="notice notice-error"><p>WooCommerce no está activado.</p></div>'; return; }

    $ref_source = isset($_GET['ref_source']) ? sanitize_text_field($_GET['ref_source']) : '';
    $start_date = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']).' 00:00:00' : '';
    $end_date   = isset($_GET['end_date'])   ? sanitize_text_field($_GET['end_date']).' 23:59:59'   : '';

    global $wpdb;

    $ref_key  = esc_sql(PS_META_REFERRER_KEY);
    $type_key = esc_sql(PS_META_SOURCE_TYPE_KEY);
    $host_sql = "LOWER(SUBSTRING_INDEX(SUBSTRING_INDEX(pm_src.meta_value, '//', -1), '/', 1))";

    $per_page = 20;
    $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
    $offset = ($current_page - 1) * $per_page;

    $total_orders = $wpdb->get_var( $wpdb->prepare("
        SELECT COUNT(DISTINCT p.ID)
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} pm_src
               ON pm_src.post_id = p.ID AND pm_src.meta_key = '{$ref_key}'
        LEFT JOIN {$wpdb->postmeta} pm_typ
               ON pm_typ.post_id = p.ID AND pm_typ.meta_key = '{$type_key}' AND pm_typ.meta_value = 'referral'
        WHERE p.post_type='shop_order'
          AND p.post_status IN ('wc-completed','wc-processing')
          AND p.post_date BETWEEN %s AND %s
          AND pm_typ.post_id IS NOT NULL
          AND {$host_sql} = %s
    ", $start_date, $end_date, strtolower($ref_source) ) );

    $orders = $wpdb->get_results( $wpdb->prepare("
        SELECT DISTINCT p.ID, p.post_date, p.post_status
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} pm_src
               ON pm_src.post_id = p.ID AND pm_src.meta_key = '{$ref_key}'
        LEFT JOIN {$wpdb->postmeta} pm_typ
               ON pm_typ.post_id = p.ID AND pm_typ.meta_key = '{$type_key}' AND pm_typ.meta_value = 'referral'
        WHERE p.post_type='shop_order'
          AND p.post_status IN ('wc-completed','wc-processing')
          AND p.post_date BETWEEN %s AND %s
          AND pm_typ.post_id IS NOT NULL
          AND {$host_sql} = %s
        ORDER BY p.post_date DESC
        LIMIT %d OFFSET %d
    ", $start_date, $end_date, strtolower($ref_source), $per_page, $offset ) );

    echo '<style>.payment-stats-table{border-collapse:separate;border-spacing:0;width:100%;border:1px solid #ccd0d4;border-radius:8px;overflow:hidden}.payment-stats-table thead{background:#f7f7f7}.payment-stats-table th,.payment-stats-table td{padding:12px 15px;text-align:left}.payment-stats-table tbody tr:nth-child(even){background:#f9f9f9}.payment-stats-table tbody tr:hover{background:#eef6fb}.payment-stats-pagination{margin-top:20px;text-align:center}.payment-stats-pagination .page-numbers{display:inline-block;padding:8px 14px;margin:0 3px;border:1px solid #ccd0d4;border-radius:5px;background:#fff;color:#007cba;text-decoration:none;font-weight:600;transition:all .2s}.payment-stats-pagination .page-numbers:hover{background:#007cba;color:#fff}.payment-stats-pagination .page-numbers.current{background:#007cba;color:#fff;border-color:#007cba;pointer-events:none}</style>';

    echo '<div class="wrap"><h1>Órdenes con Referral: '.esc_html($ref_source).' ('.intval($total_orders).' órdenes)</h1>';
    echo '<table class="payment-stats-table"><thead><tr><th>ID de Orden</th><th>Fecha</th><th>Estado</th></tr></thead><tbody>';

    if ( $orders ) {
        foreach ( $orders as $o ) {
            echo '<tr>';
            echo '<td><a href="'.get_edit_post_link($o->ID).'">#'.esc_html($o->ID).'</a></td>';
            echo '<td>'.esc_html($o->post_date).'</td>';
            echo '<td>'.esc_html($o->post_status).'</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="3">No se encontraron órdenes para este referral.</td></tr>';
    }
    echo '</tbody></table>';

    $pagination_args = [
        'total'   => max(1, ceil($total_orders / $per_page)),
        'current' => $current_page,
        'base'    => add_query_arg('paged', '%#%'),
        'format'  => '',
    ];
    echo '<div class="payment-stats-pagination">'.paginate_links($pagination_args).'</div></div>';
}
