<?php
/**
 * WAOLM - Cron de migración automática
 * Archivo: waolm_migrate_cron.php
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
define( 'TIEMPO_ENTRE_CRON', 600 ); // 5 minutos = 300
// 1) Intervalo personalizado: cada 5 minutos (300s)
add_filter('cron_schedules', function ($schedules) {
    if (!isset($schedules['waolm_every_five_minutes'])) {
        $schedules['waolm_every_five_minutes'] = array(
            'interval' => TIEMPO_ENTRE_CRON, // 5 minutos = 300
            'display'  => __('WAOLM: cada 5 minutos', 'waolm')
        );
    }
    return $schedules;
});

// 2) Hook del evento programado
add_action('waolm_migration_cron_event', 'waolm_run_migrations_cron');

/**
 * Runner del cron: ejecuta los 4 pasos de migración en orden.
 * Si migracion_final_concluida() devuelve true, se desprograma.
 */ 
function waolm_run_migrations_cron() {
    $lock_key = 'waolm_migration_cron_lock';
    if (get_transient($lock_key)) {
        return;
    }
    set_transient($lock_key, 1, TIEMPO_ENTRE_CRON);

    try {
        // Ejecuta los pasos
        if (function_exists('waolm_procesar_migracion_paso0')) {
            waolm_procesar_migracion_paso0();
        }
        if (function_exists('waolm_procesar_migracion_paso1')) {
            waolm_procesar_migracion_paso1();
        }
        if (function_exists('waolm_procesar_migracion_paso2')) {
            waolm_procesar_migracion_paso2();
        }
        if (function_exists('waolm_procesar_migracion_paso3')) {
            //waolm_procesar_migracion_paso3(); 
        }

        // Si la migración ya está finalizada, limpiamos el cron
        if (function_exists('migracion_final_concluida') && migracion_final_concluida()) {
            wp_clear_scheduled_hook('waolm_migration_cron_event');
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[WAOLM] Migración finalizada. Cron desactivado.');
            }
        }

    } catch (\Throwable $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[WAOLM] Error en waolm_run_migrations_cron: ' . $e->getMessage());
        }
    } finally {
        delete_transient($lock_key);
    }
}

// 3) Programación inicial: primer disparo a los 5 minutos
register_activation_hook(__FILE__, function () {
    wp_clear_scheduled_hook('waolm_migration_cron_event');
    $timestamp = time() + TIEMPO_ENTRE_CRON;
    wp_schedule_event($timestamp, 'waolm_every_five_minutes', 'waolm_migration_cron_event');
});

// 4) Salvaguarda: si por alguna razón se desprogramó y aún no está finalizada, reprograma
add_action('init', function () {
    if (
        !wp_next_scheduled('waolm_migration_cron_event') &&
        (!function_exists('migracion_final_concluida') || !migracion_final_concluida())
    ) {
        $timestamp = time() + TIEMPO_ENTRE_CRON;
        wp_schedule_event($timestamp, 'waolm_every_five_minutes', 'waolm_migration_cron_event');
    }
});
